

#include <time.h>
#include <stdlib.h>
#include <math.h>
#include"rfc_amalgamated.h"
#include"lice/lice/lice.h"

#define IMG_WIDTH 340
#define IMG_HEIGHT 285

// animation data
#define MAX_BALL_RADIUS 8
#define SPEED_RANGE 2
#define ALPHA_DECAY 50
#define ALPHA_INC 0.1
#define PARTICLE_COUNT  80
#define BLUR_AMOUNT 9

#define LINES_COUNT 6

#define FRAME_TIME 59

int GetRandomNumber()
{
	return rand() % 255;
}

float GetRandomFloat()
{
	return GetRandomNumber() / (float)255;
}

class Particle
{
public:
	int size, fadeBack;
	float alpha, x, y, xSpeed, ySpeed, alphaDecay;

	Particle()
	{
		x = GetRandomFloat() * IMG_WIDTH;
		y = GetRandomFloat() * IMG_HEIGHT;

		alpha = GetRandomFloat();
		size = GetRandomNumber() % MAX_BALL_RADIUS;

		xSpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;
		ySpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;
		alphaDecay = GetRandomFloat() / ALPHA_DECAY;

		fadeBack = 0;
	}

	void updateParticle()
	{
		x += xSpeed;
		y += ySpeed;

		alpha -= alphaDecay;

		if (alpha <= 0)
		{
			x = GetRandomFloat() * IMG_WIDTH;
			y = GetRandomFloat() * IMG_HEIGHT;

			xSpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;
			ySpeed = (GetRandomFloat() * (SPEED_RANGE * 2)) - SPEED_RANGE;

			fadeBack = 1;
		}


		if (fadeBack == 1)
		{
			alpha += ALPHA_INC;

			if (alpha >= 1)
			{
				alpha = 1;
				fadeBack = 0;
			}
		}


	}
};

class Line
{
public:

	float angle, centerY, range, xspeed, yspeed, xpos, ypos;
	float drawingY;
	int startY, centerPasses;
	int imageWidth;

	void initLine(int _startY)
	{
		startY = _startY;
		range = 50;
		xspeed = 6;
		yspeed = 0.04;
		centerY = IMG_HEIGHT / 2;
		centerPasses = 0;

		imageWidth = IMG_WIDTH;
		angle = yspeed*(startY - centerY);
		xpos = 0;
		ypos = startY;
		drawingY = startY;
	}

	void updateLine()
	{
		xpos += xspeed;
		angle += yspeed;
		ypos = centerY + sin(angle) * range;
	}

	void drawLine(LICE_IBitmap *canvas)
	{
		updateLine();
		xpos = 0;
		while (xpos<imageWidth)
		{
			int x1 = xpos;
			int y1 = ypos;

			updateLine();
			LICE_Line(canvas, x1, y1, xpos, ypos, LICE_RGBA(255, 255, 255, 255), 1.0, LICE_BLIT_MODE_COPY, true);
		}

		drawingY = drawingY + 1;

		if (drawingY == startY + (range * 3) + 8)
		{
			drawingY = startY;
		}

		angle = yspeed*(drawingY - centerY);
	}

};


class NOWComponent
{
protected:
	bool compVisible;
	int compX, compY;
	int compWidth, compHeight;


public:
	NOWComponent()
	{
		compX = 0;
		compY = 0;
		compVisible = true;
		compWidth = 0;
		compHeight = 0;
	}

	virtual void SetPosition(int x, int y)
	{
		compX = x;
		compY = y;
	}

	virtual int GetPosX()
	{
		return compX;
	}

	virtual int GetPosY()
	{
		return compY;
	}

	virtual void SetSize(int width, int height)
	{
		compWidth = width;
		compHeight = height;
	}

	virtual int GetWidth()
	{
		return compWidth;
	}

	virtual int GetHeight()
	{
		return compHeight;
	}

	virtual void SetVisible(bool visible)
	{
		compVisible = visible;
	}

	virtual bool IsVisible()
	{
		return compVisible;
	}


	virtual bool IsMouseWithinArea(int x, int y)
	{
		int relX = x - compX;
		int relY = y - compY;

		if ((0<relX) && (relX <= compWidth) && (0<relY) && (relY <= compHeight))
		{
			return true;
		}
		return false;
	}

	//events!

	virtual void OnMouseLDown(int xpos, int ypos){}

	virtual void OnMouseRDown(int xpos, int ypos){}

	virtual void OnMouseLUp(int xpos, int ypos){}

	virtual void OnMouseRUp(int xpos, int ypos){}

	virtual void OnMouseMove(int xpos, int ypos, WPARAM fwKeys){}

	virtual void OnMouseOver(WPARAM fwKeys){}

	virtual void OnMouseLost(WPARAM fwKeys){}

	virtual void OnPaint(LICE_IBitmap *canvas){}

	virtual ~NOWComponent(){}

};

class NOWImage : public NOWComponent
{
protected:
	LICE_IBitmap *image;

public:
	NOWImage()
	{
		image = 0;
	}

	virtual void LoadFromFile(char* fileName)
	{
		image = LICE_LoadPNG(fileName);
		if (image)
		{
			compWidth = image->getWidth();
			compHeight = image->getHeight();
		}
	}

	virtual void OnPaint(LICE_IBitmap *canvas)
	{
		LICE_Blit(canvas, image, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
	}

	virtual ~NOWImage()
	{
		if (image)
			delete image;
	}
};

class NOWButton : public NOWComponent
{
protected:
	LICE_IBitmap *normImage,*downImage;
	bool isOver, isDown;

public:
	NOWButton()
	{
		normImage = 0;
		downImage = 0;
		isOver = false;
		isDown = false;
	}

	virtual void LoadFromFile(char* normImagePath, char* downImagePath)
	{
		normImage = LICE_LoadPNG(normImagePath);
		if (normImage)
		{
			compWidth = normImage->getWidth();
			compHeight = normImage->getHeight();

			downImage = LICE_LoadPNG(downImagePath);
		}
	}

	virtual void OnPaint(LICE_IBitmap *canvas)
	{
		if (isDown)
			LICE_Blit(canvas, downImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);
		else
			LICE_Blit(canvas, normImage, compX, compY, 0, 0, compWidth, compHeight, 1.0, LICE_BLIT_USE_ALPHA);	
	}

	void OnButtonClick()
	{
		
	}

	virtual bool IsMouseWithinArea(int x, int y)
	{
		int relX = x - compX;
		int relY = y - compY;

		if ((0<relX) && (relX <= compWidth) && (0<relY) && (relY <= compHeight))
		{
			LICE_pixel p = LICE_GetPixel(isDown ? downImage : normImage, relX, relY);
			if (LICE_GETA(p) > 100)
				return true;
		}
		return false;
	}

	void OnMouseOver(WPARAM fwKeys)
	{
		isOver = true;
		if (fwKeys == MK_LBUTTON)
			isDown = true;
	}

	void OnMouseLost(WPARAM fwKeys)
	{
		isOver = false;
		if (isDown)
			isDown = false;
	}

	void OnMouseLDown(int xpos, int ypos)
	{
		isDown = true;
	}

	void OnMouseLUp(int xpos, int ypos)
	{
		bool prevDown = isDown;
		isDown = false;

		if (prevDown)
		{
			OnButtonClick();
		}
	}

	virtual ~NOWButton()
	{
		if (normImage)
			delete normImage;

		if (downImage)
			delete downImage;
	}

};

class MainWindow : public KWindow, public KThread
{
private:
	LICE_SysBitmap *backBuffer;
	LICE_IBitmap *backPanel, *gradImage, *ballImage, *linesImage;
	LICE_IBitmap *shapeImg, *maskImg, *glassImg;

	Particle *particles[PARTICLE_COUNT];
	Line* lines[LINES_COUNT];

	// for fps control
	double pcFreq;

	// client area drag data
	bool windowDraging;
	int cld_mouse_x;
	int cld_mouse_y;

	KPointerList<NOWComponent*> *componentList;
	NOWComponent *mouseOverComponent;

	NOWButton *playBtn;

	int starHideDelay,upperStarPos,upperStarStepIndex;
	int downStarPos;
	bool showFirstHalf;

	LICE_IBitmap *starsImg;
public:

	

	void DrawWindow()
	{
		// clear back buffer and draw window shape image
		LICE_FillRect(backBuffer, 0, 0, IMG_WIDTH, IMG_HEIGHT, LICE_RGBA(255, 255, 255, 0), 0.0F, LICE_BLIT_MODE_COPY);
		LICE_Blit(backBuffer, shapeImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);
		
		// draw grad img on back panel
		LICE_Blit(backPanel, gradImage, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_MODE_COPY);

		// draw balls on back panel
		LICE_FillRect(ballImage, 0, 0, IMG_WIDTH, IMG_HEIGHT, LICE_RGBA(255, 255, 255, 0), 1.0, LICE_BLIT_MODE_COPY);
		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			particles[i]->updateParticle();
			if (particles[i]->size)
				LICE_FillCircle(ballImage, particles[i]->x, particles[i]->y, particles[i]->size, LICE_RGBA(255, 255, 255, 255), particles[i]->alpha, LICE_BLIT_MODE_COPY, true);
		}
		for (int i = 0; i<BLUR_AMOUNT; i++)
		{
			LICE_Blur(ballImage, ballImage, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT);
		}
		LICE_Blit(backPanel, ballImage, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);

		// draw lines on back panel
		for (int i = 0; i<LINES_COUNT; i++)
		{
			lines[i]->drawLine(linesImage);
		}
		for (int i = 0; i<14; i++)
		{
			LICE_Blur(linesImage, linesImage, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT);
		}
		LICE_Blit(backPanel, linesImage, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);

		// draw ui components on back panel	
		for (int i = 0; i < componentList->GetSize(); i++)
		{
			NOWComponent* component = componentList->GetPointer(i);
			if (component->IsVisible())
				component->OnPaint(backPanel);
		}


		// draw stars
		if (starHideDelay>0)
		{
			starHideDelay--;
		}else // show upper & down star
		{
			int starPattern[96] = { -1, 6, 5, 4, 3, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 1, 1, 0, 0, 0, 1, 1, 2, 2, 4, 5, 6, -1,
				-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 
				-1, -1, -1, -1, -1, -1, -1,-1, -1, 6, 5, 4, 3, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 2, 2, 1, 1, 0, 0, 0, 1, 1, 2, 2, 4, 5, 6, -1 };

			if ((showFirstHalf & (upperStarStepIndex < 48)) || ((!showFirstHalf)&(48 < upperStarStepIndex)))
			{

				if (starPattern[upperStarStepIndex] != -1) // -1 means do not show
					LICE_Blit(backPanel, starsImg, upperStarPos, 10, 0, 17 + (starPattern[upperStarStepIndex] * 70), 68, 70, 1.0, LICE_BLIT_USE_ALPHA);


				if (starPattern[upperStarStepIndex] != -1) // -1 means do not show
					LICE_Blit(backPanel, starsImg, downStarPos, 170, 0, 17 + (starPattern[upperStarStepIndex] * 70), 68, 70, 1.0, LICE_BLIT_USE_ALPHA);
			}

			upperStarPos += 3;
			upperStarStepIndex++;

			downStarPos -= 3;

			if (upperStarStepIndex == 96) // star came to end
			{
				upperStarPos = -5;
				upperStarStepIndex = 0;

				downStarPos = 280;

				starHideDelay = 100 + (GetRandomNumber() % 100 * 2);
				showFirstHalf = (GetRandomNumber() % 2)==1;
			}
		}


		// draw glass panel on back panel
		LICE_Blit(backPanel, glassImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);

		// draw mask image on back panel and clear pink area
		LICE_Blit(backPanel, maskImg, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);
		LICE_SetAlphaFromColorMask(backPanel, LICE_RGBA(255, 0, 255, 0));

		// draw back panel on backbuffer
		LICE_Blit(backBuffer, backPanel, 0, 0, 0, 0, IMG_WIDTH, IMG_HEIGHT, 1.0, LICE_BLIT_USE_ALPHA);


		// Create memory DC
		HDC hdcScreen = GetDC(NULL);
		HDC hDC = CreateCompatibleDC(hdcScreen);

		// Create memory bitmap
		HBITMAP hBmp = CreateCompatibleBitmap(hdcScreen, IMG_WIDTH, IMG_HEIGHT);
		HBITMAP hBmpOld = (HBITMAP)SelectObject(hDC, hBmp);

		BitBlt(hDC, 0, 0, compWidth, compHeight, backBuffer->getDC(), 0, 0, SRCCOPY);

		// Call UpdateLayeredWindow
		BLENDFUNCTION blend = { 0 };
		blend.BlendOp = AC_SRC_OVER;
		blend.SourceConstantAlpha = 255;
		blend.AlphaFormat = AC_SRC_ALPHA;
		SIZE szWnd = { IMG_WIDTH, IMG_HEIGHT };
		POINT ptSrc = { 0, 0 };
		UpdateLayeredWindow(compHWND, hdcScreen, NULL, &szWnd, hDC, &ptSrc, 0, &blend, ULW_ALPHA);

		SelectObject(hDC, hBmpOld);
		DeleteObject(hBmp);
		DeleteDC(hDC);
		ReleaseDC(NULL, hdcScreen);
	}

	void Run() // drawing thread
	{
		
		while (!threadShouldStop)
		{
			LARGE_INTEGER li;
			QueryPerformanceCounter(&li);
			__int64 counterStart = li.QuadPart;

			DrawWindow();

			QueryPerformanceCounter(&li);
			double deltaTime = double(li.QuadPart - counterStart) / pcFreq;

			/*char buff[256];
			sprintf(buff, "%d", (int)deltaTime);
			MessageBoxA(0, buff, buff, 0);*/

			if ((FRAME_TIME - deltaTime)>0)
				Sleep(FRAME_TIME - deltaTime);
		}
		isThreadRunning = false;
	}

	void init()
	{
		LARGE_INTEGER li;
		QueryPerformanceFrequency(&li);
		pcFreq = double(li.QuadPart) / 1000.0;

		starHideDelay = 50 ;
		upperStarPos = -5;
		upperStarStepIndex = 0;
		downStarPos = 280;
		showFirstHalf = 1;

		// initialize drag data!
		windowDraging = false;

		componentList = new KPointerList<NOWComponent*>();
		mouseOverComponent = 0;

		backBuffer = new LICE_SysBitmap(IMG_WIDTH, IMG_HEIGHT);
		backPanel = new LICE_MemBitmap(IMG_WIDTH, IMG_HEIGHT);
		gradImage = new LICE_MemBitmap(IMG_WIDTH, IMG_HEIGHT);
		ballImage = new LICE_MemBitmap(IMG_WIDTH, IMG_HEIGHT);
		linesImage = new LICE_MemBitmap(IMG_WIDTH, IMG_HEIGHT);

		shapeImg = LICE_LoadPNG("c:\\shape.png");
		maskImg = LICE_LoadPNG("c:\\mask.png");
		glassImg = LICE_LoadPNG("c:\\glasspanel.png");
		starsImg = LICE_LoadPNG("c:\\star.png");

		// generate grad img
		int h, s, v;
		LICE_RGB2HSV(117, 86, 250, &h, &s, &v);
		for (int i = 0; i<IMG_HEIGHT; i++)
		{
			int newV = (i / (float)IMG_HEIGHT)*(float)v;
			int r, g, b;
			LICE_HSV2RGB(h, s, newV, &r, &g, &b);
			LICE_Line(gradImage, 0, i, IMG_WIDTH, i, LICE_RGBA(r, g, b, 255), 1.0, LICE_BLIT_MODE_COPY, true);
		}

		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			particles[i] = new Particle();
		}

		int linePatterns[LINES_COUNT] = { 204, 238, 218, 210, 227, 203 };
		for (int i = 0; i<LINES_COUNT; i++)
		{
			lines[i] = new Line();
			lines[i]->initLine(linePatterns[i]);
		}
		linesImage = new LICE_SysBitmap(IMG_WIDTH, IMG_HEIGHT);
		LICE_FillRect(linesImage, 0, 0, IMG_WIDTH, IMG_HEIGHT, LICE_RGBA(255, 255, 255, 0), 1.0, LICE_BLIT_MODE_COPY);


		playBtn = new NOWButton();
		playBtn->LoadFromFile("c:\\play_btn_norm.png", "c:\\play_btn_down.png");
		playBtn->SetPosition(50, 50);
		AddNowComponent(playBtn);

		SetStyle(WS_POPUP | WS_SYSMENU | WS_MINIMIZEBOX);
		SetExStyle(WS_EX_LAYERED);
		SetSize(IMG_WIDTH, IMG_HEIGHT);
		CreateComponent();
		DrawWindow();
		CenterScreen();

		this->StartThread(); // start animation drawing
	}

	void OnClose()
	{
		this->ThreadShouldStop();

		while (isThreadRunning){} //wait till animation drawing thread finish

		::DestroyWindow(compHWND);
	}

	void OnPaint() // handle random dummy paint msgs
	{
		PAINTSTRUCT ps;
		BeginPaint(compHWND, &ps);

		EndPaint(compHWND, &ps);
	}

	void AddNowComponent(NOWComponent *component)
	{
		if (component)
			componentList->AddPointer(component);
	}

	NOWComponent* GetComponentAt(int xPos, int yPos)
	{
		for (int i = componentList->GetSize()-1; i>=0; i--) // search top to bottom
		{
			NOWComponent* component = componentList->GetPointer(i);
			if (component->IsVisible())
			{
				if (component->IsMouseWithinArea(xPos, yPos))
				{
					return component;
				}
			}
		}
		return 0;
	}

	void OnMouseLDown(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);

		if (component) // LDown on component!
		{
			component->OnMouseLDown(xPos, yPos); // send event!
		}
		else // LDown for client area drag
		{
			OnMLButtonDownForDrag(xPos, yPos);
		}
	}

	void OnMouseMove(int xPos, int yPos, WPARAM fwKeys)
	{
		if (windowDraging) // client area drag!
		{
			OnMouseMoveForDrag();
		}
		else // send event to appropriate component!
		{
			NOWComponent* component = GetComponentAt(xPos, yPos);

			if (component)// mouse is on component!
			{
				if (component == mouseOverComponent)// mouse is still on prev comp!
				{
					component->OnMouseMove(xPos, yPos, fwKeys);
				}
				else // mouse is on new component!
				{
					if (mouseOverComponent)// tell it to prev comp!
						mouseOverComponent->OnMouseLost(fwKeys);

					mouseOverComponent = component;// new mouse over comp!
					mouseOverComponent->OnMouseOver(fwKeys);
				}
			}
			else // mouse is not on component!
			{
				if (mouseOverComponent)// tell it to prev comp!
				{
					mouseOverComponent->OnMouseLost(fwKeys);
					mouseOverComponent = 0;
				}
			}
		}
	}

	void OnMouseLUp(int xPos, int yPos)
	{
		if (windowDraging) // client area drag!
		{
			OnMLButtonUpForDrag();
		}
		else // send event to appropriate component!
		{
			NOWComponent* component = GetComponentAt(xPos, yPos);

			if (component)// mouse is on component!
			{
				component->OnMouseLUp(xPos, yPos);
			}

		}
	}

	void OnMouseRDown(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);
		if (component)
			component->OnMouseRDown(xPos, yPos);
	}

	void OnMouseRUp(int xPos, int yPos)
	{
		NOWComponent* component = GetComponentAt(xPos, yPos);
		if (component)
			component->OnMouseRUp(xPos, yPos);
	}

	LRESULT WindowProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
	{
		switch (msg)
		{
		case WM_PAINT:
			this->OnPaint();
			break;
		case WM_ERASEBKGND:
			return 1;
		case WM_MOUSEMOVE:
			this->OnMouseMove(LOWORD(lParam), HIWORD(lParam), wParam);
			break;
		case WM_LBUTTONDOWN:
			this->OnMouseLDown(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_LBUTTONUP:
			this->OnMouseLUp(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_RBUTTONDOWN:
			this->OnMouseRDown(LOWORD(lParam), HIWORD(lParam));
			break;
		case WM_RBUTTONUP:
			this->OnMouseRUp(LOWORD(lParam), HIWORD(lParam));
			break;
		default:
			return KWindow::WindowProc(hwnd, msg, wParam, lParam);
		}

		return 0;
	}

	void OnMLButtonDownForDrag(int xPos, int yPos)
	{
		windowDraging = true;
		SetCapture(compHWND);

		cld_mouse_x = xPos;
		cld_mouse_y = yPos;
	}

	void OnMouseMoveForDrag()
	{
		POINT pos;
		GetCursorPos(&pos);
		SetPosition(pos.x - cld_mouse_x, pos.y - cld_mouse_y);
	}

	void OnMLButtonUpForDrag()
	{
		ReleaseCapture();
		windowDraging = false;
	}

	void releaseResources()
	{
		delete backBuffer;
		delete gradImage;
		delete backPanel;
		delete shapeImg;
		delete maskImg;
		delete ballImage;
		delete glassImg;
		delete linesImage;
		delete starsImg;

		for (int i = 0; i<PARTICLE_COUNT; i++)
		{
			delete particles[i];
		}

		for (int i = 0; i<LINES_COUNT; i++)
		{
			delete lines[i];
		}


		for (int i = 0; i < componentList->GetSize(); i++)
			delete componentList->GetPointer(i);

		delete componentList;

	}
};

class NOWPlayer : public KApplication
{
public:
	int Main(KString **argv, int argc)
	{

		MainWindow mainWindow;

		srand(time(NULL));

		mainWindow.init();
		mainWindow.SetVisible(true);

		DoMessagePump(false);

		mainWindow.releaseResources();

		return 0;
	}
};

START_RFC_APPLICATION(NOWPlayer)